import 'dart:convert';
import 'package:easy_localization/easy_localization.dart';
import 'package:firebase_messaging/firebase_messaging.dart';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:zcart_delivery/models/user_model.dart';
import 'package:zcart_delivery/network/api.dart';
import 'package:zcart_delivery/network/network_helper.dart';
import 'package:zcart_delivery/translations/locale_keys.g.dart';

final userProvider = FutureProvider<UserModel?>((ref) async {
  try {
    final response =
        await NetworkHelper.getRequest(API.profile, bearerToken: true);

    if (response != null) {
      debugPrint(response.body);
      if (isRequestSuccessful(response.statusCode)) {
        final user = UserModel.fromJson(json.decode(response.body)["data"]);
        return user;
      } else if (response.statusCode == 401) {
        Fluttertoast.showToast(msg: LocaleKeys.you_are_not_logged_in.tr());

        return null;
      } else {
        Fluttertoast.showToast(
            msg: jsonDecode(response.body)['message'] ??
                LocaleKeys.something_went_wrong.tr());
        return null;
      }
    } else {
      Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
      return null;
    }
  } catch (e) {
    Fluttertoast.showToast(msg: e.toString());
    return null;
  }
});

class AuthController {
  static Future<UserModel?> login(String email, String password) async {
    String? fcmToken = await getFCMToken();

    Map<String, String> requestBody = {
      'email': email,
      'password': password,
      if (fcmToken != null) 'fcm_token': fcmToken,
    };

    try {
      final response = await NetworkHelper.postRequest(API.login, requestBody);

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          Fluttertoast.showToast(msg: LocaleKeys.login_success.tr());

          return UserModel.fromJson(jsonDecode(response.body)['data']);
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());
          return null;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return null;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return null;
    }
  }

  static Future<bool> logout() async {
    try {
      final response =
          await NetworkHelper.postRequest(API.logout, {}, bearerToken: true);

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          Fluttertoast.showToast(msg: LocaleKeys.logout_success.tr());
          await setAccessToken('');
          return true;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());
          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }

  static Future<bool> updateProfile(UserModel user) async {
    try {
      final response = await NetworkHelper.postRequest(
        API.profile,
        user.toMap(),
        bearerToken: true,
      );

      print("Response: ${response?.body}");

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          return true;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());

          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }

  static Future<bool> updatePassword(String oldPass, String newPass) async {
    Map<String, String> requestBody = {
      'oldpassword': oldPass,
      'newpassword': newPass,
    };
    try {
      final response = await NetworkHelper.postRequest(
          API.updatePassword, requestBody,
          bearerToken: true);

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          if (jsonDecode(response.body)['isSuccess'] == true) {
            Fluttertoast.showToast(msg: jsonDecode(response.body)['message']);
          }
          return jsonDecode(response.body)['isSuccess'] ?? false;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());

          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }

  static Future<bool> forgotPassword(String email) async {
    Map<String, String> requestBody = {
      'email': email,
    };
    try {
      final response =
          await NetworkHelper.postRequest(API.forgotPassword, requestBody);

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'],
              toastLength: Toast.LENGTH_LONG);
          return true;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());

          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }

  static Future<bool> validateCode(String token) async {
    Map<String, String> requestBody = {
      'token': token,
    };
    try {
      final response = await NetworkHelper.getRequest(API.resetPassword,
          requestBody: requestBody);

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          return true;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());

          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }

  static Future<bool> resetPassword(String password, String token) async {
    Map<String, String> requestBody = {
      'password': password,
      'token': token,
    };
    try {
      final response =
          await NetworkHelper.postRequest(API.resetPassword, requestBody);

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          Fluttertoast.showToast(msg: jsonDecode(response.body)['message']);
          return true;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());

          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }
}

Future<String?> getFCMToken() async {
  return await FirebaseMessaging.instance.getToken();
}
